#!/bin/bash
# LIMAN SDK Fetch, Build, and Distribution Script
#
# This handy script downloads a fresh copy of the LIMAN SDK package into a temporary
# folder (or fetches from a local repository of packages), builds the SDK with your
# official license file, and then copies the built library to a target destination.
# (c) Baroks Inc.
#
# Usage: ./lg_fetch_build_distrib_sdk.sh [options] [platform] [output_dir]
#   platform: linux64x86 | osx64x86 | osx64arm | mingw64 | mingw32 (default: auto-detect)
#   output_dir: where to place built SDK (default: /tmp/liman_licensed_$$)
#
# Options:
#   -f, --force          Actually execute the build (default is dry-run mode)
#   -h, --help           Show this help message
#   --local              Use local release directory instead of downloading from web
#   --target-dir=DIR     Copy built libliman_sdk.a to specified directory
#
# Environment Variables:
#   LICMAKER_LICENSE_FILE_C  - Path to your official license .c file (required)
#                              This file will be copied to license/usr_liman_lic.c
#                              and compiled into the SDK library
#
# Example:
#   LICMAKER_LICENSE_FILE_C=/path/to/usr_liman_lic.c ./lg_fetch_build_distrib_sdk.sh -f
#   LICMAKER_LICENSE_FILE_C=/path/to/usr_liman_lic.c ./lg_fetch_build_distrib_sdk.sh -f --local
#   LICMAKER_LICENSE_FILE_C=/path/to/usr_liman_lic.c ./lg_fetch_build_distrib_sdk.sh -f --target-dir=/opt/myapp/lib
#

set -e

# Save current directory and resolve script directory
ORIG_DIR="$(pwd)"
SCRIPT_DIR="$(cd "$(dirname "$0")" && pwd)"

show_usage() {
    echo "Usage: $0 [options] [platform] [output_dir]"
    echo ""
    echo "Downloads LIMAN SDK, builds it with your license file, and optionally"
    echo "copies the built library to a target directory."
    echo ""
    echo "Arguments:"
    echo "  platform        Platform: linux64x86 | osx64x86 | osx64arm | mingw64 | mingw32"
    echo "                  (default: auto-detect)"
    echo "  output_dir      Temporary build directory (default: /tmp/liman_licensed_\$\$)"
    echo ""
    echo "Options:"
    echo "  -f, --force          Actually execute the build (default is dry-run mode)"
    echo "  -h, --help           Show this help message"
    echo "  --local              Use local release directory instead of downloading"
    echo "  --target-dir=DIR     Copy built library and headers to specified directory"
    echo ""
    echo "Environment Variables:"
    echo "  LICMAKER_LICENSE_FILE_C  Path to your license .c file"
    echo "                           (default: ${SCRIPT_DIR}/../license/usr_liman_lic.c)"
    echo ""
    echo "Examples:"
    echo "  $0 -f"
    echo "  $0 -f --local"
    echo "  $0 -f --target-dir=/opt/myapp"
    echo "  LICMAKER_LICENSE_FILE_C=/path/to/lic.c $0 -f linux64x86"
    exit 0
}

# Show usage if no arguments
if [ $# -eq 0 ]; then
    show_usage
fi

# Cleanup function to remove temp directory
cleanup() {
    if [ -n "${OUTPUT_DIR}" ] && [ -d "${OUTPUT_DIR}" ]; then
        echo "Cleaning up temp directory: ${OUTPUT_DIR}"
        rm -rf "${OUTPUT_DIR}"
    fi
}

# Set default license file if not specified
LICMAKER_LICENSE_FILE_C="${LICMAKER_LICENSE_FILE_C:-${SCRIPT_DIR}/../license/usr_liman_lic.c}"

# Read version from liman.h if available
LIMAN_H="${SCRIPT_DIR}/../include/liman.h"
if [ -f "${LIMAN_H}" ]; then
    LIMVER_MAJOR=$(grep -E "^#define LIMVER_MAJOR" "${LIMAN_H}" | awk '{print $3}')
    LIMVER_MINOR=$(grep -E "^#define LIMVER_MINOR" "${LIMAN_H}" | awk '{print $3}')
else
    LIMVER_MAJOR=1
    LIMVER_MINOR=3
fi
LIMAN_VERSION="${LIMVER_MAJOR}.${LIMVER_MINOR}"

BASE_URL="https://baroks.com/Downloads"
LOCAL_RELEASE_DIR="$HOME/dev/c/liman/liman-release/release"

# Auto-detect platform
detect_platform() {
    local os=$(uname -s)
    local arch=$(uname -m)
    case "$os" in
        Linux*)
            case "$arch" in
                aarch64|arm64) echo "linux64arm";;
                *) echo "linux64x86";;
            esac
            ;;
        Darwin*)
            case "$arch" in
                arm64) echo "osx64arm";;
                *) echo "osx64x86";;
            esac
            ;;
        MINGW64*|MSYS_NT*) echo "mingw64";;
        MINGW32*) echo "mingw32";;
        CYGWIN*) echo "mingw64";;
        *) echo "linux64x86";;
    esac
}

# Parse arguments
USE_LOCAL=0
PLATFORM=""
OUTPUT_DIR=""
TARGET_DIR=""
FORCE=0

for arg in "$@"; do
    case "$arg" in
        -h|--help)
            show_usage
            ;;
        -f|--force)
            FORCE=1
            ;;
        --local|-local)
            USE_LOCAL=1
            ;;
        --target-dir=*)
            TARGET_DIR="${arg#*=}"
            # Expand ~ to home directory
            TARGET_DIR="${TARGET_DIR/#\~/$HOME}"
            ;;
        -*)
            echo "Unknown option: $arg"
            echo "Use -h for help."
            exit 1
            ;;
        *)
            if [ -z "$PLATFORM" ]; then
                PLATFORM="$arg"
            elif [ -z "$OUTPUT_DIR" ]; then
                OUTPUT_DIR="$arg"
            fi
            ;;
    esac
done

PLATFORM="${PLATFORM:-$(detect_platform)}"
OUTPUT_DIR="${OUTPUT_DIR:-/tmp/liman_licensed_$$}"
PACKAGE="liman_core-${LIMAN_VERSION}-${PLATFORM}.tar.gz"
DOWNLOAD_URL="${BASE_URL}/${PACKAGE}"

echo "========================================"
echo "LIMAN SDK Licensed Build Script"
echo "========================================"
echo "Platform:     ${PLATFORM}"
echo "Package:      ${PACKAGE}"
echo "Output dir:   ${OUTPUT_DIR}"
echo "License file: ${LICMAKER_LICENSE_FILE_C}"
if [ -n "${TARGET_DIR}" ]; then
    echo "Target dir:   ${TARGET_DIR}"
fi
echo "Force:        ${FORCE}"
echo "========================================"
echo

# Check for license file
if [ ! -f "${LICMAKER_LICENSE_FILE_C}" ]; then
    echo "ERROR: License file not found: ${LICMAKER_LICENSE_FILE_C}"
    echo ""
    echo "Set LICMAKER_LICENSE_FILE_C environment variable to your license file path."
    exit 1
fi

# Dry-run mode check
if [ "${FORCE}" -eq 0 ]; then
    echo "*** DRY-RUN MODE ***"
    echo ""
    echo "This is a preview of what would be executed."
    echo "Use -f or --force to actually run the build."
    echo ""
    echo "Would execute:"
    echo "  1. Download/copy package: ${PACKAGE}"
    echo "  2. Extract to: ${OUTPUT_DIR}"
    echo "  3. Install license: ${LICMAKER_LICENSE_FILE_C}"
    echo "  4. Build SDK library"
    if [ -n "${TARGET_DIR}" ]; then
        echo "  5. Copy to: ${TARGET_DIR}/lib/ and ${TARGET_DIR}/include/"
    fi
    echo "  6. Cleanup temp directory"
    echo ""
    echo "Run with -f to execute."
    exit 0
fi

# Set up cleanup trap only when actually executing
trap cleanup EXIT

# Step 1: Download or copy package
echo "[Step 1] Obtaining LIMAN SDK package..."
mkdir -p "${OUTPUT_DIR}"
cd "${OUTPUT_DIR}"

LOCAL_PACKAGE="${LOCAL_RELEASE_DIR}/${PACKAGE}"
if [ "${USE_LOCAL}" -eq 1 ]; then
    if [ -f "${LOCAL_PACKAGE}" ]; then
        echo "Copying from local release: ${LOCAL_PACKAGE}"
        cp "${LOCAL_PACKAGE}" "${PACKAGE}"
    else
        echo "ERROR: Local package not found: ${LOCAL_PACKAGE}"
        exit 1
    fi
else
    echo "Downloading from ${DOWNLOAD_URL}..."
    if command -v curl &> /dev/null; then
        curl -L -o "${PACKAGE}" "${DOWNLOAD_URL}"
    elif command -v wget &> /dev/null; then
        wget -O "${PACKAGE}" "${DOWNLOAD_URL}"
    else
        echo "ERROR: Neither curl nor wget found. Cannot download."
        exit 1
    fi
fi

if [ ! -f "${PACKAGE}" ]; then
    echo "ERROR: Package not found!"
    exit 1
fi
echo "Package ready: $(ls -lh ${PACKAGE})"
echo

# Step 2: Extract package
echo "[Step 2] Extracting package..."
tar -xzf "${PACKAGE}"
EXTRACT_DIR="liman_core-${LIMAN_VERSION}-${PLATFORM}"

if [ ! -d "${EXTRACT_DIR}" ]; then
    echo "ERROR: Extracted directory ${EXTRACT_DIR} not found!"
    exit 1
fi
echo "Extracted to: ${EXTRACT_DIR}"
echo

# Step 3: Copy license file
echo "[Step 3] Installing license file..."
echo "Working directory: $(pwd)"
LICENSE_DEST="${EXTRACT_DIR}/license/usr_liman_lic.c"
cp -v "${LICMAKER_LICENSE_FILE_C}" "${LICENSE_DEST}"

# Also create the _copy.c file that some Makefiles expect
cp -v "${LICMAKER_LICENSE_FILE_C}" "${EXTRACT_DIR}/license/usr_liman_lic_copy.c"
echo "License installed."
echo

# Step 4: Navigate to lib/<platform> and build
echo "[Step 4] Building libliman_sdk..."
cd "${EXTRACT_DIR}/lib/${PLATFORM}"
pwd

# Determine make command
if [[ "${PLATFORM}" == mingw* ]]; then
    MAKE_CMD="mingw32-make"
else
    MAKE_CMD="make"
fi

# Build the SDK library
echo "Running: ${MAKE_CMD}"
${MAKE_CMD}
echo

# Step 5: Verify build
echo "[Step 5] Verifying build..."

# Check for static library
if [[ "${PLATFORM}" == win* ]]; then
    LIB_FILE="libliman_sdk.lib"
else
    LIB_FILE="libliman_sdk.a"
fi

if [ -f "${LIB_FILE}" ]; then
    echo "SUCCESS: ${LIB_FILE} built"
    ls -lh "${LIB_FILE}"
else
    echo "ERROR: ${LIB_FILE} not found!"
    exit 1
fi

# Check for tools
if [[ "${PLATFORM}" == mingw* ]] || [[ "${PLATFORM}" == win* ]]; then
    LIMUTIL="./limutil.exe"
    LIMGEN="./limgen.exe"
else
    LIMUTIL="./limutil"
    LIMGEN="./limgen"
fi

if [ -f "${LIMUTIL}" ]; then
    echo "limutil: $(ls -lh ${LIMUTIL})"
fi
if [ -f "${LIMGEN}" ]; then
    echo "limgen: $(ls -lh ${LIMGEN})"
fi
echo

# Copy to target directory if specified
if [ -n "${TARGET_DIR}" ]; then
    echo "[Step 6] Copying library to target directory..."
    echo "Source: $(pwd)/${LIB_FILE}"
    echo "Target: ${TARGET_DIR}/lib/"
    ls -la "${LIB_FILE}"
    mkdir -p "${TARGET_DIR}/lib"
    cp -v "${LIB_FILE}" "${TARGET_DIR}/lib/"
    ls -la "${TARGET_DIR}/lib/${LIB_FILE}"
    # Also copy liman.h to include directory
    mkdir -p "${TARGET_DIR}/include"
    cp -v "${LIMAN_H}" "${TARGET_DIR}/include/"
    echo
fi

# Summary
echo "========================================"
echo "Licensed SDK Build Complete!"
echo "========================================"
echo "Platform:      ${PLATFORM}"
if [ -n "${TARGET_DIR}" ]; then
    echo "Library:       ${TARGET_DIR}/lib/${LIB_FILE}"
    echo "Header:        ${TARGET_DIR}/include/liman.h"
fi
echo "========================================"
