#!/bin/bash
# Download and Reinstall LIMAN SDK Script
# Downloads LIMAN SDK from baroks.com (or local release), compares versions, and updates if newer
# (c) Baroks Inc.
#
# Usage: ./lg_download_reinstall.sh [options] <install_dir>
#   install_dir: SDK installation directory (REQUIRED)
#
# Options:
#   -p, --platform=PLATFORM  Platform: linux64x86 | linux64arm | osx64x86 | osx64arm | mingw64 | mingw32 (default: auto-detect)
#   -f, --force              Force install even if version check fails or versions are equal
#   -l, --local              Use local release directory instead of downloading from web
#   -h, --help               Show this help message
#

set -e

SCRIPT_DIR="$(cd "$(dirname "$0")" && pwd)"
BASE_URL="https://baroks.com/Downloads"
LOCAL_RELEASE_DIR="$HOME/dev/c/liman/liman-release/release"

# Auto-detect platform
detect_platform() {
    local os=$(uname -s)
    local arch=$(uname -m)
    case "$os" in
        Linux*)
            case "$arch" in
                aarch64|arm64) echo "linux64arm";;
                *) echo "linux64x86";;
            esac
            ;;
        Darwin*)
            case "$arch" in
                arm64) echo "osx64arm";;
                *) echo "osx64x86";;
            esac
            ;;
        MINGW64*|MSYS_NT*) echo "mingw64";;
        MINGW32*) echo "mingw32";;
        CYGWIN*) echo "mingw64";;
        *) echo "linux64x86";;
    esac
}

show_usage() {
    echo "Usage: $0 [options] <install_dir>"
    echo ""
    echo "Downloads LIMAN SDK and installs/updates to the specified directory."
    echo ""
    echo "Arguments:"
    echo "  install_dir              SDK installation directory (REQUIRED)"
    echo ""
    echo "Options:"
    echo "  -p, --platform=PLATFORM  Platform: linux64x86 | linux64arm | osx64x86 | osx64arm | mingw64 | mingw32"
    echo "                           (default: auto-detect)"
    echo "  -f, --force              Force install even if version check fails or"
    echo "                           local version is same/newer than remote"
    echo "  -l, --local              Use local release directory instead of downloading"
    echo "  -h, --help               Show this help message"
    echo ""
    echo "Examples:"
    echo "  $0 /opt/liman"
    echo "  $0 --platform=linux64x86 ~/sdk/liman"
    echo "  $0 -f /opt/liman"
    echo "  $0 -l -f ~/usr/licMaker    # Use local release packages"
    exit 1
}

# Extract version from liman.h
get_version_from_file() {
    local liman_h_file="$1"

    if [ ! -f "${liman_h_file}" ]; then
        echo "0.0.0.0"
        return
    fi

    local major=$(grep -E "^#define LIMVER_MAJOR" "${liman_h_file}" | awk '{print $3}')
    local minor=$(grep -E "^#define LIMVER_MINOR" "${liman_h_file}" | awk '{print $3}')
    local release=$(grep -E "^#define LIMVER_RELEASE" "${liman_h_file}" | awk '{print $3}')
    local rev=$(grep -E "^#define LIMVER_REVISION" "${liman_h_file}" | awk '{print $3}')

    echo "${major}.${minor}.${release}.${rev}"
}

# Compare versions (return 0 if v1 < v2, 1 otherwise)
version_less_than() {
    local v1="$1"
    local v2="$2"

    # Convert to comparable format
    local v1_cmp=$(echo "$v1" | awk -F. '{printf "%03d%03d%03d%03d", $1, $2, $3, $4}')
    local v2_cmp=$(echo "$v2" | awk -F. '{printf "%03d%03d%03d%03d", $1, $2, $3, $4}')

    [ "$v1_cmp" -lt "$v2_cmp" ]
}

# Parse arguments
PLATFORM=""
INSTALL_DIR=""
FORCE=0
USE_LOCAL=0

while [ $# -gt 0 ]; do
    case "$1" in
        -h|--help)
            show_usage
            ;;
        -f|--force)
            FORCE=1
            shift
            ;;
        -l|--local)
            USE_LOCAL=1
            shift
            ;;
        -p)
            PLATFORM="$2"
            shift 2
            ;;
        --platform=*)
            PLATFORM="${1#*=}"
            shift
            ;;
        -*)
            echo "ERROR: Unknown option: $1"
            show_usage
            ;;
        *)
            INSTALL_DIR="$1"
            shift
            ;;
    esac
done

# Validate required argument
if [ -z "${INSTALL_DIR}" ]; then
    echo "ERROR: install_dir is required"
    echo ""
    show_usage
fi

# Default platform if not specified
PLATFORM="${PLATFORM:-$(detect_platform)}"
WORK_DIR="/tmp/liman_download_$$"

# Resolve to absolute path (create if doesn't exist for new installs)
if [ ! -d "${INSTALL_DIR}" ]; then
    mkdir -p "${INSTALL_DIR}"
fi
INSTALL_DIR="$(cd "${INSTALL_DIR}" && pwd)"

# Safety check: prevent overwriting development folder
if [ -f "${INSTALL_DIR}/scripts/test_customer_experience.sh" ]; then
    echo "ERROR: Target directory appears to be the LIMAN SDK development folder."
    echo "       Found: ${INSTALL_DIR}/scripts/test_customer_experience.sh"
    echo ""
    echo "This script is intended for installing the SDK to a deployment location,"
    echo "not for updating the development source tree."
    echo ""
    echo "To update the development folder, use git pull instead."
    rm -rf "${WORK_DIR}" 2>/dev/null
    exit 1
fi

INSTALLED_LIMAN_H="${INSTALL_DIR}/include/liman.h"

echo "========================================"
echo "LIMAN SDK Download & Update"
echo "========================================"
echo "Platform:     ${PLATFORM}"
echo "Install dir:  ${INSTALL_DIR}"
echo "Force:        ${FORCE}"
echo "Use local:    ${USE_LOCAL}"
echo ""

# Get local version from installed SDK
LOCAL_VERSION=$(get_version_from_file "${INSTALLED_LIMAN_H}")
echo "Local version: ${LOCAL_VERSION}"

if [ "${LOCAL_VERSION}" = "0.0.0.0" ]; then
    echo "NOTE: No existing SDK found - will perform fresh install."
fi

# Get package to check version
echo ""
mkdir -p "${WORK_DIR}"
cd "${WORK_DIR}"

# Read version from trunk's liman.h if available (for package name)
TRUNK_LIMAN_H="${SCRIPT_DIR}/../include/liman.h"
if [ -f "${TRUNK_LIMAN_H}" ]; then
    PKG_MAJOR=$(grep -E "^#define LIMVER_MAJOR" "${TRUNK_LIMAN_H}" | awk '{print $3}')
    PKG_MINOR=$(grep -E "^#define LIMVER_MINOR" "${TRUNK_LIMAN_H}" | awk '{print $3}')
else
    PKG_MAJOR=$(echo "$LOCAL_VERSION" | cut -d. -f1)
    PKG_MINOR=$(echo "$LOCAL_VERSION" | cut -d. -f2)
fi
PACKAGE="liman_core-${PKG_MAJOR}.${PKG_MINOR}-${PLATFORM}.tar.gz"

if [ "${USE_LOCAL}" -eq 1 ]; then
    echo "Copying package from local release..."
    LOCAL_PACKAGE="${LOCAL_RELEASE_DIR}/${PACKAGE}"
    if [ -f "${LOCAL_PACKAGE}" ]; then
        cp "${LOCAL_PACKAGE}" "${PACKAGE}"
        echo "Copied: ${LOCAL_PACKAGE}"
    else
        echo "ERROR: Local package not found: ${LOCAL_PACKAGE}"
        rm -rf "${WORK_DIR}"
        exit 1
    fi
else
    echo "Downloading package to check version..."
    DOWNLOAD_URL="${BASE_URL}/${PACKAGE}"
    if command -v curl &> /dev/null; then
        curl --connect-timeout 15 --max-time 300 -f -L -o "${PACKAGE}" "${DOWNLOAD_URL}" 2>&1 | grep -E "HTTP|Length" || true
    elif command -v wget &> /dev/null; then
        wget --timeout=15 -O "${PACKAGE}" "${DOWNLOAD_URL}" 2>&1 | grep -E "HTTP|Length" || true
    else
        echo "ERROR: Neither curl nor wget found."
        rm -rf "${WORK_DIR}"
        exit 1
    fi
fi

if [ ! -f "${PACKAGE}" ] || [ ! -s "${PACKAGE}" ]; then
    echo "ERROR: Package not available!"
    rm -rf "${WORK_DIR}"
    exit 1
fi

# Extract and get remote version
echo "Extracting to check version..."
tar -xzf "${PACKAGE}"
EXTRACT_DIR="liman_core-${PKG_MAJOR}.${PKG_MINOR}-${PLATFORM}"
REMOTE_LIMAN_H="${EXTRACT_DIR}/include/liman.h"

if [ ! -f "${REMOTE_LIMAN_H}" ]; then
    echo "ERROR: Cannot find liman.h in downloaded package!"
    rm -rf "${WORK_DIR}"
    exit 1
fi

REMOTE_MAJOR=$(grep -E "^#define LIMVER_MAJOR" "${REMOTE_LIMAN_H}" | awk '{print $3}')
REMOTE_MINOR=$(grep -E "^#define LIMVER_MINOR" "${REMOTE_LIMAN_H}" | awk '{print $3}')
REMOTE_RELEASE=$(grep -E "^#define LIMVER_RELEASE" "${REMOTE_LIMAN_H}" | awk '{print $3}')
REMOTE_REV=$(grep -E "^#define LIMVER_REVISION" "${REMOTE_LIMAN_H}" | awk '{print $3}')
REMOTE_VERSION="${REMOTE_MAJOR}.${REMOTE_MINOR}.${REMOTE_RELEASE}.${REMOTE_REV}"

echo "Remote version: ${REMOTE_VERSION}"
echo ""

# Compare versions
SHOULD_UPDATE=0
if version_less_than "${LOCAL_VERSION}" "${REMOTE_VERSION}"; then
    echo "*** Newer version available! ***"
    echo "Current: ${LOCAL_VERSION}"
    echo "Latest:  ${REMOTE_VERSION}"
    SHOULD_UPDATE=1
elif [ "${FORCE}" -eq 1 ]; then
    echo "Force mode enabled - will update regardless of version."
    echo "Current: ${LOCAL_VERSION}"
    echo "Remote:  ${REMOTE_VERSION}"
    SHOULD_UPDATE=1
fi

if [ "${SHOULD_UPDATE}" -eq 1 ]; then
    echo ""

    # Prompt for confirmation (skip if force mode)
    if [ "${FORCE}" -eq 0 ]; then
        read -p "Update to newer version? (Y/n): " -n 1 -r
        echo
        if [[ $REPLY =~ ^[Nn]$ ]]; then
            echo "Update cancelled."
            rm -rf "${WORK_DIR}"
            exit 0
        fi
    fi

    echo ""
    echo "Updating SDK files..."

    # Backup existing files as compressed tar.gz
    BACKUP_FILE="${INSTALL_DIR}/../liman_backup_$(date +%Y%m%d_%H%M%S).tar.gz"
    echo "Creating backup at: ${BACKUP_FILE}"
    # Build list of items to backup (only those that exist)
    BACKUP_ITEMS=""
    for item in "${EXTRACT_DIR}"/*; do
        item_name=$(basename "$item")
        if [ -e "${INSTALL_DIR}/${item_name}" ]; then
            BACKUP_ITEMS="${BACKUP_ITEMS} ${item_name}"
        fi
    done
    if [ -n "${BACKUP_ITEMS}" ]; then
        tar -czf "${BACKUP_FILE}" -C "${INSTALL_DIR}" ${BACKUP_ITEMS} 2>/dev/null || true
    fi

    # Copy all new files (overwrite existing)
    echo "Copying new SDK files..."
    for item in "${EXTRACT_DIR}"/*; do
        item_name=$(basename "$item")
        if [ -d "$item" ]; then
            # Directory - merge contents
            mkdir -p "${INSTALL_DIR}/${item_name}"
            cp -rf "$item"/* "${INSTALL_DIR}/${item_name}/"
            echo "  Updated: ${item_name}/"
        else
            # File - copy directly
            cp -f "$item" "${INSTALL_DIR}/"
            echo "  Updated: ${item_name}"
        fi
    done

    echo ""
    echo "========================================"
    echo "Update complete!"
    echo "========================================"
    echo "Updated: ${LOCAL_VERSION} -> ${REMOTE_VERSION}"
    echo "Backup:  ${BACKUP_FILE}"
    echo "========================================"
else
    echo "Local version is up to date or newer."
    echo "No update needed."
    if [ "${FORCE}" -eq 0 ]; then
        echo "Use -f to force reinstall."
    fi
fi

# Cleanup
echo ""
echo "Cleaning up temporary files..."
rm -rf "${WORK_DIR}"
echo "Done."
