@echo off
setlocal enabledelayedexpansion
rem Download and Reinstall LIMAN SDK Script (Windows)
rem Downloads LIMAN SDK from baroks.com (or local release), compares versions, and updates if newer
rem (c) Baroks Inc.
rem
rem Usage: lg_download_reinstall.bat [options] <install_dir>
rem   install_dir: SDK installation directory (REQUIRED)
rem
rem Options:
rem   -p PLATFORM  Platform: win32x86 | win64x86 | mingw64 | mingw32 (default: win64x86)
rem   -f           Force install even if version check fails or versions are equal
rem   -l           Use local release directory instead of downloading from web
rem   -h           Show this help message
rem

set SCRIPT_DIR=%~dp0
rem Remove trailing backslash
set SCRIPT_DIR=%SCRIPT_DIR:~0,-1%
set BASE_URL=https://baroks.com/Downloads
set LOCAL_RELEASE_DIR=%USERPROFILE%\dev\c\liman\liman-release\release

rem Default values
set PLATFORM=win64x86
set INSTALL_DIR=
set FORCE=0
set USE_LOCAL=0

rem Parse arguments
:parse_args
if "%~1"=="" goto done_args
if /i "%~1"=="-h" goto show_usage
if /i "%~1"=="--help" goto show_usage
if /i "%~1"=="-f" (
    set FORCE=1
    shift
    goto parse_args
)
if /i "%~1"=="--force" (
    set FORCE=1
    shift
    goto parse_args
)
if /i "%~1"=="-l" (
    set USE_LOCAL=1
    shift
    goto parse_args
)
if /i "%~1"=="--local" (
    set USE_LOCAL=1
    shift
    goto parse_args
)
if /i "%~1"=="-p" (
    set PLATFORM=%~2
    shift
    shift
    goto parse_args
)
rem Check for --platform=VALUE
set "_ARG=%~1"
if /i "!_ARG:~0,11!"=="--platform=" (
    set "PLATFORM=!_ARG:~11!"
    shift
    goto parse_args
)
rem Not an option, must be install_dir
if "%INSTALL_DIR%"=="" (
    set "INSTALL_DIR=%~f1"
)
shift
goto parse_args

:done_args

rem Validate required argument
if "%INSTALL_DIR%"=="" (
    echo ERROR: install_dir is required
    echo.
    goto show_usage
)

rem Create install directory if it doesn't exist
if not exist "%INSTALL_DIR%" mkdir "%INSTALL_DIR%"

rem Safety check: prevent overwriting development folder
if exist "%INSTALL_DIR%\scripts\test_customer_experience.sh" (
    echo ERROR: Target directory appears to be the LIMAN SDK development folder.
    echo        Found: %INSTALL_DIR%\scripts\test_customer_experience.sh
    echo.
    echo This script is intended for installing the SDK to a deployment location,
    echo not for updating the development source tree.
    echo.
    echo To update the development folder, use git pull instead.
    exit /b 1
)
if exist "%INSTALL_DIR%\scripts\test_customer_experience.bat" (
    echo ERROR: Target directory appears to be the LIMAN SDK development folder.
    echo        Found: %INSTALL_DIR%\scripts\test_customer_experience.bat
    echo.
    echo This script is intended for installing the SDK to a deployment location,
    echo not for updating the development source tree.
    echo.
    echo To update the development folder, use git pull instead.
    exit /b 1
)

set INSTALLED_LIMAN_H=%INSTALL_DIR%\include\liman.h
set WORK_DIR=%TEMP%\liman_download_%RANDOM%

echo ========================================
echo LIMAN SDK Download ^& Update
echo ========================================
echo Platform:     %PLATFORM%
echo Install dir:  %INSTALL_DIR%
echo Force:        %FORCE%
echo Use local:    %USE_LOCAL%
echo.

rem Get local version from installed SDK
set LOCAL_VERSION=0.0.0.0
set LOCAL_MAJOR=0
set LOCAL_MINOR=0
if exist "%INSTALLED_LIMAN_H%" (
    for /f "tokens=3" %%a in ('findstr /R /C:"^#define LIMVER_MAJOR" "%INSTALLED_LIMAN_H%"') do set LOCAL_MAJOR=%%a
    for /f "tokens=3" %%a in ('findstr /R /C:"^#define LIMVER_MINOR" "%INSTALLED_LIMAN_H%"') do set LOCAL_MINOR=%%a
    for /f "tokens=3" %%a in ('findstr /R /C:"^#define LIMVER_RELEASE" "%INSTALLED_LIMAN_H%"') do set LOCAL_RELEASE=%%a
    for /f "tokens=3" %%a in ('findstr /R /C:"^#define LIMVER_REVISION" "%INSTALLED_LIMAN_H%"') do set LOCAL_REV=%%a
    set LOCAL_VERSION=!LOCAL_MAJOR!.!LOCAL_MINOR!.!LOCAL_RELEASE!.!LOCAL_REV!
)
echo Local version: %LOCAL_VERSION%

if "%LOCAL_VERSION%"=="0.0.0.0" (
    echo NOTE: No existing SDK found - will perform fresh install.
)

rem Get package to check version
echo.
if exist "%WORK_DIR%" rmdir /s /q "%WORK_DIR%"
mkdir "%WORK_DIR%"
cd /d "%WORK_DIR%"

rem Read version from trunk's liman.h if available (for package name)
set TRUNK_LIMAN_H=%SCRIPT_DIR%\..\include\liman.h
set PKG_MAJOR=1
set PKG_MINOR=3
if exist "%TRUNK_LIMAN_H%" (
    for /f "tokens=3" %%a in ('findstr /R /C:"^#define LIMVER_MAJOR" "%TRUNK_LIMAN_H%"') do set PKG_MAJOR=%%a
    for /f "tokens=3" %%a in ('findstr /R /C:"^#define LIMVER_MINOR" "%TRUNK_LIMAN_H%"') do set PKG_MINOR=%%a
) else (
    if not "%LOCAL_MAJOR%"=="0" set PKG_MAJOR=%LOCAL_MAJOR%
    if not "%LOCAL_MINOR%"=="0" set PKG_MINOR=%LOCAL_MINOR%
)
set PACKAGE=liman_core-%PKG_MAJOR%.%PKG_MINOR%-%PLATFORM%.tar.gz

if "%USE_LOCAL%"=="1" (
    echo Copying package from local release...
    set LOCAL_PACKAGE=%LOCAL_RELEASE_DIR%\%PACKAGE%
    if exist "!LOCAL_PACKAGE!" (
        copy "!LOCAL_PACKAGE!" "%PACKAGE%" >nul
        echo Copied: !LOCAL_PACKAGE!
    ) else (
        echo ERROR: Local package not found: !LOCAL_PACKAGE!
        cd /d "%INSTALL_DIR%"
        rmdir /s /q "%WORK_DIR%"
        exit /b 1
    )
) else (
    echo Downloading package to check version...
    set DOWNLOAD_URL=%BASE_URL%/%PACKAGE%
    where curl >nul 2>&1
    if !ERRORLEVEL!==0 (
        curl --connect-timeout 15 --max-time 300 -f -L -o "%PACKAGE%" "!DOWNLOAD_URL!"
    ) else (
        powershell -Command "Invoke-WebRequest -TimeoutSec 15 -Uri '!DOWNLOAD_URL!' -OutFile '%PACKAGE%'"
    )
)

if not exist "%PACKAGE%" (
    echo ERROR: Package not available!
    cd /d "%INSTALL_DIR%"
    rmdir /s /q "%WORK_DIR%"
    exit /b 1
)

rem Extract and get remote version
echo Extracting to check version...
tar -xzf "%PACKAGE%"
set EXTRACT_DIR=liman_core-%PKG_MAJOR%.%PKG_MINOR%-%PLATFORM%
set REMOTE_LIMAN_H=%EXTRACT_DIR%\include\liman.h

if not exist "%REMOTE_LIMAN_H%" (
    echo ERROR: Cannot find liman.h in downloaded package!
    cd /d "%INSTALL_DIR%"
    rmdir /s /q "%WORK_DIR%"
    exit /b 1
)

for /f "tokens=3" %%a in ('findstr /R /C:"^#define LIMVER_MAJOR" "%REMOTE_LIMAN_H%"') do set REMOTE_MAJOR=%%a
for /f "tokens=3" %%a in ('findstr /R /C:"^#define LIMVER_MINOR" "%REMOTE_LIMAN_H%"') do set REMOTE_MINOR=%%a
for /f "tokens=3" %%a in ('findstr /R /C:"^#define LIMVER_RELEASE" "%REMOTE_LIMAN_H%"') do set REMOTE_RELEASE=%%a
for /f "tokens=3" %%a in ('findstr /R /C:"^#define LIMVER_REVISION" "%REMOTE_LIMAN_H%"') do set REMOTE_REV=%%a
set REMOTE_VERSION=%REMOTE_MAJOR%.%REMOTE_MINOR%.%REMOTE_RELEASE%.%REMOTE_REV%

echo Remote version: %REMOTE_VERSION%
echo.

rem Compare versions
set SHOULD_UPDATE=0
call :version_less_than "%LOCAL_VERSION%" "%REMOTE_VERSION%"
if %ERRORLEVEL%==0 (
    echo *** Newer version available! ***
    echo Current: %LOCAL_VERSION%
    echo Latest:  %REMOTE_VERSION%
    set SHOULD_UPDATE=1
) else if "%FORCE%"=="1" (
    echo Force mode enabled - will update regardless of version.
    echo Current: %LOCAL_VERSION%
    echo Remote:  %REMOTE_VERSION%
    set SHOULD_UPDATE=1
)

if "%SHOULD_UPDATE%"=="0" (
    echo Local version is up to date or newer.
    echo No update needed.
    if "%FORCE%"=="0" echo Use -f to force reinstall.
    goto cleanup
)

rem Prompt for confirmation (skip if force mode)
if "%FORCE%"=="0" (
    set /p REPLY="Update to newer version? (Y/n): "
    if /i "!REPLY!"=="n" goto cancelled
    if /i "!REPLY!"=="no" goto cancelled
)

echo.
echo Updating SDK files...

rem Backup existing files as compressed tar.gz
for /f "tokens=1-3 delims=/ " %%a in ('echo %DATE%') do set TODAY=%%c%%a%%b
for /f "tokens=1-2 delims=:. " %%a in ('echo %TIME%') do set NOW=%%a%%b
set TODAY=!TODAY: =0!
set NOW=!NOW: =0!
set BACKUP_FILE=%INSTALL_DIR%\..\liman_backup_!TODAY!_!NOW!.tar.gz
echo Creating backup at: !BACKUP_FILE!
cd /d "%INSTALL_DIR%"
rem Build list of items to backup (only those that exist in source package)
set BACKUP_ITEMS=
cd /d "%WORK_DIR%\%EXTRACT_DIR%"
for /d %%D in (*) do (
    if exist "%INSTALL_DIR%\%%D" set BACKUP_ITEMS=!BACKUP_ITEMS! %%D
)
for %%F in (*.*) do (
    if not "%%~xF"=="" if exist "%INSTALL_DIR%\%%F" set BACKUP_ITEMS=!BACKUP_ITEMS! %%F
)
cd /d "%INSTALL_DIR%"
if not "!BACKUP_ITEMS!"=="" tar -czf "!BACKUP_FILE!" !BACKUP_ITEMS! >nul 2>&1

rem Copy all new files (overwrite existing)
echo Copying new SDK files...
cd /d "%WORK_DIR%\%EXTRACT_DIR%"
for /d %%D in (*) do (
    if not exist "%INSTALL_DIR%\%%D" mkdir "%INSTALL_DIR%\%%D"
    xcopy /E /I /Y /Q "%%D" "%INSTALL_DIR%\%%D\" >nul
    echo   Updated: %%D\
)
for %%F in (*.*) do (
    if not "%%~xF"=="" (
        copy /Y "%%F" "%INSTALL_DIR%\" >nul
        echo   Updated: %%F
    )
)

echo.
echo ========================================
echo Update complete!
echo ========================================
echo Updated: %LOCAL_VERSION% -^> %REMOTE_VERSION%
echo Backup:  !BACKUP_FILE!
echo ========================================
goto cleanup

:cancelled
echo Update cancelled.
goto cleanup

:cleanup
rem Cleanup
echo.
echo Cleaning up temporary files...
cd /d "%INSTALL_DIR%"
rmdir /s /q "%WORK_DIR%" 2>nul
echo Done.
goto :eof

:show_usage
echo Usage: %~nx0 [options] ^<install_dir^>
echo.
echo Downloads LIMAN SDK and installs/updates to the specified directory.
echo.
echo Arguments:
echo   install_dir              SDK installation directory (REQUIRED)
echo.
echo Options:
echo   -p PLATFORM              Platform: win32x86 ^| win64x86 ^| mingw64 ^| mingw32
echo                            (default: win64x86)
echo   -f                       Force install even if version check fails or
echo                            local version is same/newer than remote
echo   -l                       Use local release directory instead of downloading
echo   -h                       Show this help message
echo.
echo Examples:
echo   %~nx0 C:\sdk\liman
echo   %~nx0 -p win64x86 C:\sdk\liman
echo   %~nx0 -f C:\sdk\liman
echo   %~nx0 -l -f C:\sdk\liman    (Use local release packages)
exit /b 1

rem Function to compare versions (return 0 if v1 < v2, 1 otherwise)
:version_less_than
setlocal
set V1=%~1
set V2=%~2

rem Parse v1
for /f "tokens=1 delims=." %%a in ("%V1%") do set V1_1=%%a
for /f "tokens=2 delims=." %%a in ("%V1%") do set V1_2=%%a
for /f "tokens=3 delims=." %%a in ("%V1%") do set V1_3=%%a
for /f "tokens=4 delims=." %%a in ("%V1%") do set V1_4=%%a

rem Parse v2
for /f "tokens=1 delims=." %%a in ("%V2%") do set V2_1=%%a
for /f "tokens=2 delims=." %%a in ("%V2%") do set V2_2=%%a
for /f "tokens=3 delims=." %%a in ("%V2%") do set V2_3=%%a
for /f "tokens=4 delims=." %%a in ("%V2%") do set V2_4=%%a

rem Pad with zeros
set V1_1=000%V1_1%
set V1_1=%V1_1:~-3%
set V1_2=000%V1_2%
set V1_2=%V1_2:~-3%
set V1_3=000%V1_3%
set V1_3=%V1_3:~-3%
set V1_4=000%V1_4%
set V1_4=%V1_4:~-3%

set V2_1=000%V2_1%
set V2_1=%V2_1:~-3%
set V2_2=000%V2_2%
set V2_2=%V2_2:~-3%
set V2_3=000%V2_3%
set V2_3=%V2_3:~-3%
set V2_4=000%V2_4%
set V2_4=%V2_4:~-3%

set V1_CMP=%V1_1%%V1_2%%V1_3%%V1_4%
set V2_CMP=%V2_1%%V2_2%%V2_3%%V2_4%

if %V1_CMP% LSS %V2_CMP% (
    endlocal & exit /b 0
) else (
    endlocal & exit /b 1
)

endlocal
