/** \example limutil.c
 * \brief
 * This is a utility application to perform various tasks with LIMAN SDK:
 *    - Generate keypairs
 *    - Determine hardware and users features (macid,userid,cpuid,etc..)
 *    - Calculate hash of input data (files or strings)
 *    - Hide small data in BMP (bitmap) files.
 *    - UUID generation
 */
#include "liman.h"
#include "getopt_long.c"

#if 1
// Created by 'make' either from 'usr_liman_lic.c' or 'usr_liman_lic_mock.c'
#include "usr_liman_lic_copy.c"
#endif

static struct option long_options[] =
{
   {"hostid",     required_argument,  0, 'H', "MASK",         "Display hostid w.r.t specified bitmask '%s'"
                           "\n                                    (MASK & 1)    - cpuid"
                           "\n                                    (MASK & 2)    - diskid"
                           "\n                                    (MASK & 4)    - hostname"
                           "\n                                    (MASK & 8)    - userid" 
                           "\n                                    (MASK & 16)   - dmi/wmic" 
                           "\n                                    (MASK & 32)   - MACid (1st)"
                           "\n                                    (MASK & 64)   - MACid_2 (2nd)"
                           "\n                                    (MASK & 128)  - MACid_3 (3rd)"
                           "\n                                    (MASK & 256)  - MACid_4 (4th)"
   },
   {"macid",      required_argument,  0, 'M', "N",            "Display the id of the %s'th mac adapter. N=0..NMAX"},
   {"userid",     no_argument,        0, 'u', NULL,           "Display userid for current user"},
   {"dmi",        no_argument,        0, 'd', NULL,           "Display DMI/UUID for current host"},
   {"cpuid",      no_argument,        0, 'C', NULL,           "Display cpuid of current host"},
   {"keysize",    required_argument,  0, 'k', "KEYSIZE",      "Specify '%s' as the keysize for the keypair to be generated"},
   {"seed",       required_argument,  0, 's', "SEED",         "Specify '%s' as the seed (if reproducibility of keys is required)"},   
   {"genkey",     no_argument,        0, 'g', NULL,           "Generate a random product-id and a keypair"}, //
   {"genuuid",    no_argument,        0, 'D', NULL,           "Generate a random uuid"}, //
   {"product",    required_argument,  0, 'P', "PRODUCT_CODE", "Specify '%s' as the product code for the generated keypair"}, //
   {"gethash",    required_argument,  0, 'a', "FILE",         "Calculate and display the hash value of file '%s'"}, //
   {"pubkeyfile", required_argument,  0, 'U', "PUBKEYF",      "Specify '%s' as the public key file"},
   {"privkeyfile",required_argument,  0, 'I', "PRIVKEYF",     "Specify '%s' as the private key file"},
   {"sysinfo",    no_argument,        0, 'S', NULL,           "Generate encrypted system info to be sent to vendor"},   
   {"errlist",    no_argument,        0, 'e', NULL,           "Error list"}, //
   {"stencode",   required_argument,  0, 'y', "TEXT",         "Hide specified '%s' in a BMP file"}, //
   {"bmpfile",    required_argument,  0, 'b', "FILE",         "BMP file to encode/decode text '%s'"}, //
   {"c2lic",      required_argument,  0, 't', "key_lic.c",    "Convert '%s' to 'key.lic'"}, //
   {"help",       no_argument,        0, 'h', NULL,           "Display usage"},
   {0, 0, 0, 0, 0, 0}
};

//
//
//
void usage(void) {
  struct option *opt;
  char strbuf[255], argdesc[1024];
  opt = long_options;
  fprintf(stdout,"\nUsage: limutil [options]\n");
  while (opt->name) {
    if (opt->has_arg!=no_argument) {
      sprintf(argdesc,opt->descr,opt->argval);
      sprintf(strbuf,"--%s=%s",opt->name,opt->argval);
    } else {
      sprintf(strbuf,"--%s",opt->name);
      strcpy(argdesc,opt->descr);
    }
    fprintf(stdout,"\n    -%c, %-22s  %s",opt->val,strbuf,argdesc);
    opt++;    
    //printf("\n%x, %s, desc=%s argval=%s",opt,opt->name,opt->descr,opt->argval);fflush(stdout);
  }
  fprintf(stdout,"\n\n");
  fprintf(stdout,"Examples:\n");
  fprintf(stdout,"         $ limutil --genkey --keysize=%d --seed=4321\n\n", LIMPAR_DEFAULT_KEYSIZE);
  fprintf(stdout,"           Generates a keypair of size %d-bits and a random product-id \n"
                 "           using seed value 4321\n\n", LIMPAR_DEFAULT_KEYSIZE);
  fprintf(stdout,"         $ limutil --genkey --keysize=1024 --product=ABCDEF01\n\n");
  fprintf(stdout,"           Creates a generates a keypair of size 1024-bits \n"
                 "           to be associated with product-id 'ABCDEF01'\n");
  fprintf(stdout,"Remark:\n");
  fprintf(stdout,"          Generated keys are written into the folder specified by the\n"
                 "          'product-id'\n");
  return;
}

//
//
//
int main (int argc, char **argv)
{
  int i, c, iexit=0;
  char szVerInfo[256];
  /* getopt_long stores the option index here. */
  int option_index = 0, macid_num=0, isGenKey=0;

  lim_get_version_info(szVerInfo);
  fprintf(stdout, "limutil - LIMAN SDK %s\n", szVerInfo);
  char szPublicKeyFile[LIMPAR_MAX_PATH]="";
  char szPrivateKeyFile[LIMPAR_MAX_PATH]="";
  char *szProductIdPtr=NULL;  
  char szProductId[LIMPAR_MAX_LEN_PRODUCTID+1]="";
  char hostid[LIMPAR_MAX_LEN_HOSTID+1]="";
  char macid[255];
  char cpuid[255];
  char userid[LIMPAR_MAX_LEN_USERID+1]="";
  char szUUID[2*LIMPAR_MAX_LEN_UUID+1]="";
  char szHash[255], szVer[255];
  int  nErr=LIM_OK, nKeySize=LIMPAR_DEFAULT_KEYSIZE, nHashSize=LIMPAR_DEFAULT_HASHSIZE, nSeed=0, isKeySize=0, isSysInfo=0, isGenUUID=0;
  pLIMENV pEnv = NULL;
  pLIMLIC pLic = NULL;
  char st_text[255];
  char bmp_file[255];
  char c_lic_file[255];
  
  st_text[0]='\0';
  bmp_file[0]='\0';
  c_lic_file[0]='\0';
  lim_get_verstr(szVer);
  fprintf(stdout,"\nLiman Version %s\n",szVer);
  if (argc<2) {
    usage();
    exit(1);
  }
  fprintf(stdout,"\n");  
  
  while (1)
  {
    c = getopt_long (argc, argv, "U:I:H:M:Cgk:s:wa:e?hSP:uDy:b:t:",
      long_options, &option_index);

    /* Detect the end of the options. */
    if (c == -1)
      break;

    switch (c)
    {
    case 0:
      /* If this option set a flag, do nothing else now. */
      if (long_options[option_index].flag != 0)
        break;
      printf ("option %s", long_options[option_index].name);
      if (optarg)
        printf (" with arg %s", optarg);
      printf ("\n");
      break;

    case 'H':
      nErr = lim_get_hostid(hostid,atoi(optarg));
      if (nErr != LIM_OK) goto ErrReturn;  
      fprintf(stdout,"hostid   : %s\n",hostid); 
      break;

    case 'u':
      nErr = lim_get_userid(userid);
      if (nErr != LIM_OK) goto ErrReturn;  
      fprintf(stdout,"userid   : %s\n",userid); 
      break;
    
    case 'U': 
      strcpy(szPublicKeyFile,optarg);
      break;

    case 'I': 
      strcpy(szPrivateKeyFile,optarg);
      break;

    case 'b': 
      strcpy(bmp_file,optarg);
      break;

    case 't': 
      strcpy(c_lic_file,optarg);
      break;

    case 'y': 
      strcpy(st_text,optarg);
      break;

    case 'M':
      macid_num=atoi(optarg);
      nErr = lim_get_macid(macid,macid_num);
      if (nErr != LIM_OK) goto ErrReturn;  
      fprintf(stdout,"macid(%d) : %s\n",macid_num,macid);       
      break;

    case 'C':
      nErr = lim_get_cpuid(cpuid);
      if (nErr != LIM_OK) goto ErrReturn;  
      fprintf(stdout,"cpuid    : %s\n",cpuid);           
      break;

    case 'D':
      isGenUUID=1;
      break;

    case 'g':      
      isGenKey=1;     
      break; 

    case 'P':
      strcpy(szProductId,optarg);
      szProductIdPtr=szProductId;
      break; 

    case 'S':
      isSysInfo=1;     
      break; 

    case 'k':
      nKeySize=atoi(optarg);
      if (nKeySize<=64) {
        nErr = LIMERR_INVALID_INPUT;  
        goto ErrReturn;
      } 
      isKeySize=1;
      break;

    case 's':
      nSeed=atoi(optarg);
      break;

    case 'e':
      for (i=LIMERR_FIRST_CARD; i<LIMERR_LAST_CARD; i++) {
        fprintf(stdout,"%s , %s\n",lim_errmacro(NULL,i),lim_errmsg(NULL,i));
      }
      break;

    case 'a':
      nErr = lim_hash_file(optarg,szHash,nHashSize);
      if (nErr) goto ErrReturn;
      fprintf(stdout,"%s\n",szHash);      
      break;  

    case '?':
    case 'h':
      /* getopt_long already printed an error message. */
      usage();
      break;

    default:
      usage();
      exit(0);
    }
  }

  if (strlen(c_lic_file)>0) {
    FILE *f=NULL;
    char *string =NULL;
    long fsize, k;
    f = fopen(c_lic_file,"r");
    if (!f) {
      fprintf(stdout,"\nError: %s\n",lim_errmsg(NULL,LIMERR_FILE_IO));
      goto ErrReturn;
    }
    fseek(f, 0, SEEK_END);
    fsize = ftell(f);
    fseek(f, 0, SEEK_SET);  /* same as rewind(f); */

    string = malloc(fsize + 1);
    if (!string) {
      fprintf(stdout,"\nError: %s\n",lim_errmsg(NULL,LIMERR_MEMORY));
      goto ErrReturn;
    }
    fread(string, 1, fsize, f);
    fclose(f);
    {
      char *ptr,*pb,*pe;
      char *token;
      ptr = strchr(string,'{');
      if (!ptr) {
        free(string);
        fprintf(stdout,"\nError: %s\n",lim_errmsg(NULL,LIMERR_INVALID_INPUT));
        goto ErrReturn;
      }
      pb = strchr(ptr+1,'{');
      pe = strchr(pb,'}');
      if (!pb || !pe) {
        free(string);
        fprintf(stdout,"\nError: %s\n",lim_errmsg(NULL,LIMERR_INVALID_INPUT));
        goto ErrReturn;
      }
      *pe='\0';

      k=0;
      f = fopen("xyz.lic","w");
      fprintf(f,"#----- BEGIN LICENSE KEY (Id:xyx) -----\n");
      token = strtok(pb+1, ",");      
      while (token) {
        if (atoi(token)>0) { fprintf(f,"%c", atoi(token));k++;}
        if (k>0 && k%64==0) fprintf(f,"\n");
        token = strtok(NULL, ",");
      }
      fprintf(f,"\n#----- END LICENSE KEY -----\n");
      fclose(f);
      fprintf(stdout,"Written .lic file to '%s'\n","xyz.lic");
    }
    goto ErrReturn;
  }
    
  if (strlen(bmp_file)>0) {
    if (strlen(st_text)>0) {
      nErr = lim_st_encode(bmp_file,st_text,bmp_file);
      if (nErr != LIM_OK) goto ErrReturn;
      fprintf(stdout, "\n");
      fprintf(stdout, "Hidden Text: '%s' is embedded into '%s'\n",st_text,bmp_file);
    } else {
      char *sz_hidden=NULL;
      sz_hidden = lim_st_decode(bmp_file,NULL,&nErr);
      fprintf(stdout, "\n");
      if (sz_hidden) {
        fprintf(stdout, "Hidden Text: '%s'\n",sz_hidden);
        if (sz_hidden) free(sz_hidden);
      } else {
        fprintf(stdout, "Error: No hidden text was found in '%s'\n",bmp_file);
        nErr = LIMERR_NOT_AVAILABLE;
        goto ErrReturn;
      }      
    } 
  } else if (strlen(st_text)>0) {
    fprintf (stdout,"Error: -b <bmp_file> is required option with -y <text>.\n");
    goto ErrReturn;
  }
  
  if (isGenKey) {
    if (!isKeySize) {
      fprintf (stdout,"Warning: Keysize not specified. Using default keysize (%d bits)\n",nKeySize);
    }
    nErr = lim_gen_keypair(nKeySize,nSeed,szProductIdPtr);
    if (nErr != LIM_OK) goto ErrReturn;
  }

  if (isKeySize) {
    if (!isGenKey) {
      fprintf (stdout,"Warning: Insufficient options.\n");
      fprintf (stdout,"Add -g (--genkey) option to generate a keypair of specified size (%d bits)\n",nKeySize);
    }
  }

  if (isSysInfo || isGenUUID) {
    // Write host-id to a file in encrypted form (typically to send it to software vendor)
#ifdef _DEBUG
    if (!isKeySize) {
      fprintf (stdout,"Warning: Keysize not specified. Using default keysize (%d bits)\n",nKeySize);
    }
#endif
    pEnv = lim_create_env(nKeySize,nHashSize,&nErr);
    if (nErr != LIM_OK) goto ErrReturn;
    if (isSysInfo) {
      // read public key to encrypt message
      nErr = lim_read_public_key(pEnv,szPublicKeyFile); 
      if (nErr != LIM_OK) goto ErrReturn;
      if (!strlen(hostid)) {
        nErr = lim_get_hostid(hostid, 0);
        if (nErr != LIM_OK) goto ErrReturn;
      }
      nErr = lim_write_secretmsg(pEnv,hostid,"usr_sysinfo.txt");
      if (nErr != LIM_OK) goto ErrReturn;
    }
    if (isGenUUID) {
      nErr = lim_gen_uuid(pEnv,nSeed,szUUID);
      if (nErr != LIM_OK) goto ErrReturn;
      fprintf(stdout,"\n%s\n",lim_canonical_uuid(szUUID));
    }
  }

  fprintf(stdout,"\n");

  /* Print any remaining command line arguments (not options). */
  if (optind < argc)
  {
    fprintf (stdout,"non-option ARGV-elements: ");
    while (optind < argc)
      fprintf (stdout,"%s ", argv[optind++]);
    fprintf(stdout,"\n");
  }
ErrReturn:
  fflush(stdout);
  if (pLic) {
    lim_free_lic(&pLic);
  }
  if (pEnv) {
    lim_free_env(&pEnv);
  }
  if (nErr!=LIM_OK) {
    iexit=1;
    fprintf(stdout,"\n");
    fprintf(stdout,"Liman Error (%d): %s\n",nErr,lim_errmsg(NULL,nErr));
  }
  exit (iexit);
}
